/* EPLAY.H
 *
 * Definitions for the EGI player DLL.
 * Copyright (c) 1996-1997, ITB CompuPhase
 */
#ifndef _EPLAY_H
#define _EPLAY_H

#if !defined(_INC_WINDOWS) && !defined(__WINDOWS_H)
#  include <windows.h>
#endif

#ifdef __cplusplus
extern "C"
{
#endif

#if !defined(_EPLAY_C)
   typedef struct {int dummy;} FAR *LPFLIC;
#  if defined(__WIN32__) || defined(_WIN32) || defined(WIN32)
#    define FLICAPI     __declspec(dllimport) __stdcall
#  else
#    define FLICAPI     WINAPI
#  endif
#endif

BOOL   FLICAPI FlicClose(LPFLIC lpFlic);
HWND   FLICAPI FlicCreateWindow(LPCSTR szCaption,DWORD dwStyle,int x,int y,
                                int width,int height,HWND hwndParent,
                                HINSTANCE hinst);
int    FLICAPI FlicError(LPFLIC lpFlic);
LPVOID FLICAPI FlicGetData(LPFLIC lpFlic,int code);
HANDLE FLICAPI FlicGetHandle(LPFLIC lpFlic,HDC hdc,int code);
int    FLICAPI FlicGetParam(LPFLIC lpFlic,int code);
BOOL   FLICAPI FlicGetRect(LPFLIC lpFlic,int code,LPRECT lpRect);
WORD   FLICAPI FlicNextFrame(LPFLIC lpFlic);
LPFLIC FLICAPI FlicOpen(LPCSTR szFilename);
void   FLICAPI FlicPaint(LPFLIC lpFlic,HDC hdc,BOOL bPaintAll);
BOOL   FLICAPI FlicPlay(LPFLIC lpFlic,HWND hwnd);
BOOL   FLICAPI FlicSetParam(LPFLIC lpFlic,int code,int value);
BOOL   FLICAPI FlicSetHandle(LPFLIC lpFlic,int code,HANDLE handle);
BOOL   FLICAPI FlicStop(LPFLIC lpFlic);

/* Parameters for FlicGetParam() and FlicSetParam() */
enum {
  FLIC_PARAM_ISPLAYING, /* 1=FLIC file is currently playing */
  FLIC_PARAM_WIDTH,
  FLIC_PARAM_HEIGHT,
  FLIC_PARAM_NUMFRAMES, /* total number of frames */
  FLIC_PARAM_CURFRAME,  /* current frame number */
  FLIC_PARAM_CONTINUOUS,/* segment is continuous (it has a ring frame) */
  FLIC_PARAM_SPEED,     /* playback speed, set to 0 for max. speed */
  FLIC_PARAM_DESIGNSPEED,/* design speed, read from FLIC file */
  FLIC_PARAM_REALSPEED, /* real playback speed */
  FLIC_PARAM_MODE,      /* BitBlt mode */
  FLIC_PARAM_LOOP,      /* loop frames or only play once */
  FLIC_PARAM_NUMSEGMENTS,
  FLIC_PARAM_CURSEGMENT,
  FLIC_PARAM_XPOS,      /* position of the animation in the window */
  FLIC_PARAM_YPOS,
  FLIC_PARAM_LABEL,     /* label of the current frame (-1 if none) */
  FLIC_PARAM_CHGPAL,    /* use current palette, don't create a new one */
  FLIC_PARAM_VERSION,   /* current version of EGI player */
  FLIC_PARAM_SEGLABEL,  /* the label of the current segment */
  FLIC_PARAM_DRAWFRAME, /* draw the frames? */
  FLIC_PARAM_FRNOTIFY,  /* notify for each frame? */
  FLIC_PARAM_MASK,      /* masks present (0=no, 1=bitmap, 2=skip-copy, 3=both) */
  FLIC_PARAM_ID,        /* user specified id */
  FLIC_PARAM_MSGNUM,    /* value of FLIC_NOTIFY */
  FLIC_PARAM_KEYFREQ,   /* key frame interval (0 if there are no key frames) */
  FLIC_PARAM_SKIPDELTAS,/* skip all delta frames up to the next key frame */
  FLIC_PARAM_PALMODE,   /* palette mode */
  FLIC_PARAM_IDENTPAL,  /* 1:1 mapping DIB colour table to logical palette? */
};

/* Values for FLIC_PARAM_MODE */
enum {
  FLIC_MODE_BITBLT,
  FLIC_MODE_DIBTODEV,
  FLIC_MODE_STRETCHDIB,
  FLIC_MODE_WING,       /* no longer supported */
  FLIC_MODE_DIBSECTION,
  FLIC_MODES
};

/* Values for FLIC_PARAM_PALMODE */
enum {
  FLIC_PAL_ADAPT,       /* adapt to system palette */
  FLIC_PAL_ANIM,        /* use the palette from the FLIC file (default) */
  FLIC_PAL_SPECIFIC,    /* use the palette passed in by FlicSetHandle() */
  FLIC_PALMODES
};

/* Parameters for FlicGetData() */
enum {
  FLIC_DATA_BITMAP,     /* pointer to the pixel data */
  FLIC_DATA_PALETTE,    /* pointer to the RGB table */
  FLIC_DATA_MASK,       /* pointer to the bitmap mask data */
  FLIC_DATA_BITMAPINFO, /* pointer to the BITMAPINFO structure */
};

/* Parameters for FlicGetHandle() and FlicSetHandle() */
enum {
  FLIC_HANDLE_PALETTE,  /* handle to the palette */
  FLIC_HANDLE_BITMAP,   /* handle to device dependent bitmap of the frame */
  FLIC_HANDLE_MASK,     /* handle to the bitmap mask */
  FLIC_HANDLE_WINGDC,   /* no longer supported (always 0) */
  FLIC_HANDLE_WINGBMP,  /* no longer supported (always 0) */
  FLIC_HANDLE_MUTEX,    /* handle to Win32 "mutex" semaphore for bitmap */
};

/* Parameters for FlicGetRect() */
enum {
  FLIC_RECT_BOUNDING,   /* bounding box of frame differences */
  FLIC_RECT_MASK,       /* bounding box of the mask */
  FLIC_RECT_FIRST,      /* first rectangle of the region of frame differences */
  FLIC_RECT_NEXT,       /* next rectangle of the region of frame differences */
};

/* Returned bits for FlicNextFrame() */
#define FLIC_FRAME_PAL      0x0001  /* palette needs updating */
#define FLIC_FRAME_IMG      0x0002  /* image needs updating, use FlicPaint() */
#define FLIC_FRAME_LAST     0x0004  /* last frame has played */
#define FLIC_FRAME_RING     0x0008  /* ring frame has played (reset to 2nd frame) */
#define FLIC_FRAME_LABEL    0x0010  /* a label was reached */
#define FLIC_FRAME_DECODED  0x8000  /* the frame was decoded correctly */

/* Returned values for FlicError() */
enum {
  FLIC_ERR_NONE,
  FLIC_ERR_FOPEN,
  FLIC_ERR_FORMAT,
  FLIC_ERR_FREAD,
  FLIC_ERR_MEMORY,
  FLIC_ERR_PARAM,
  FLIC_ERR_INVALID,
  FLIC_ERR_MUTEX,
  FLIC_ERR_UNSUPPORTED,
};

/* Message posted to the owner window when a label is reached or a segment
 * ends
 */
#define FLIC_NOTIFY     (WM_USER+98)
enum {                  /* values for wParam upon receipt of FLIC_NOTIFY */
  FN_LABEL,             /* reached a label (LOWORD(lParam)==label) */
  FN_LASTFRAME,         /* played last frame */
  FN_FRAME,             /* played (another) frame */
};


#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
class FlicWindow {
  HWND hwnd;
public:
  FlicWindow(LPCSTR szCaption,DWORD dwStyle,int x,int y,int width,int height,
             HWND Parent,HINSTANCE hinst)
  { hwnd=FlicCreateWindow(szCaption,dwStyle,x,y,width,height,Parent,hinst); }
  operator HWND() { return hwnd; }
};

class FlicAnim {
  LPFLIC lpFlic;
  // class assignment is not supported; make the operator private so
  // that the compiler won't allow the assignment operator
  FlicAnim& operator=(FlicAnim&)
    { FlicClose(lpFlic); lpFlic=NULL; return *this; }
  // copy constructor is private too (i.e. it is unsupported)
  FlicAnim(FlicAnim&)
    { lpFlic=NULL; }
public:
  FlicAnim()
    { lpFlic=NULL; }
  FlicAnim(LPCSTR szFilename)
    { lpFlic=FlicOpen(szFilename); }
  ~FlicAnim()
    { FlicClose(lpFlic); }
  // this lets you test whether the FLIC file was opened correctly
  operator int()
    { return lpFlic!=NULL; }

  BOOL Close()
    { return FlicClose(lpFlic) ? lpFlic=NULL,1 : 0; }
  int  Error()
    { return FlicError(lpFlic); }
  LPVOID GetData(int code)
    { return FlicGetData(lpFlic,code); }
  HANDLE GetHandle(HDC hdc,int code)
    { return FlicGetHandle(lpFlic,hdc,code); }
  int  GetParam(int code)
    { return FlicGetParam(lpFlic,code); }
  BOOL GetRect(int code,LPRECT lpRect)
    { return FlicGetRect(lpFlic,code,lpRect); }
  WORD NextFrame()
    { return FlicNextFrame(lpFlic); }
  BOOL Open(LPCSTR szFilename)
    { return (lpFlic=FlicOpen(szFilename))!=NULL; }
  void Paint(HDC hdc,BOOL bPaintAll)
    { FlicPaint(lpFlic,hdc,bPaintAll); }
  BOOL Play(HWND hwnd)
    { return FlicPlay(lpFlic,hwnd); }
  BOOL SetParam(int code,int value)
    { return FlicSetParam(lpFlic,code,value); }
  BOOL SetHandle(int code,HANDLE handle)
    { return FlicSetHandle(lpFlic,code,handle); }
  BOOL Stop()
    { return FlicStop(lpFlic); }
};
#endif

#endif /* _EPLAY_H */
